"""Task model describing client or regulatory tasks."""

from datetime import date
from enum import Enum
from typing import Optional
from pydantic import BaseModel, Field, validator


class Recurrence(str, Enum):
    NONE = "none"
    WEEKLY = "weekly"
    MONTHLY = "monthly"
    QUARTERLY = "quarterly"
    SEMIANNUAL = "semiannual"
    ANNUAL = "annual"


class Priority(str, Enum):
    LOW = "low"
    MEDIUM = "medium"
    HIGH = "high"


class Task(BaseModel):
    id: Optional[int] = Field(default=None)
    task_code: str = Field(..., description="Unique task code")
    title: str
    client_name: Optional[str] = None
    service_type: Optional[str] = None
    period: Optional[str] = None  # e.g., FY 2024-25, Q1, etc.
    due_date: date
    priority: Priority = Priority.MEDIUM
    recurrence: Recurrence = Recurrence.NONE
    regulatory_reference: Optional[str] = None
    notes: Optional[str] = None

    class Config:
        orm_mode = True

    @validator("due_date")
    def due_date_must_be_future(cls, v: date) -> date:
        from datetime import date as d

        today = d.today()
        # allow due today or future
        if v < today:
            raise ValueError("Due date must not be in the past.")
        return v